/*
 * Copyright (C) 2007 Freescale Semiconductor, Inc. All rights reserved.
 *
 * Author: John Rigby <jrigby@freescale.com>
 *
 * Implements the linux os dependent routines for the ppc-axe-driver
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <pthread.h>

typedef pthread_mutex_t AxeMutex_t;

#include "ppc-axe-driver.h"
#include "axe-ipc.h"

/**
 *	\brief		Get a pointer to the AXE driver mutex
 *				
 *	\return		Shared Mutex pointer
 *
 *	Returns the pointer to the mutex used by the driver code to control access
 *	to hw and sw shared resources.
 *
 *  \note		This mutex must be shared by all processes attaching to the
 *				driver.
 */
AxeMutex_t *AxeGetDriverMutex(void);

typedef pthread_cond_t AxeCond_t;

/**
 *	\brief		Get a pointer to the RX queue condition variable
 *				
 *	\return		Shared condition variable pointer
 *
 *	Returns the pointer to the condition variable used by the driver code
 *	to notify waiting threads of a new message arriving in the software
 *	RX msg queue.
 *
 *  \note		This condition variable must be shared by all processes
 *				attaching to the driver.
 */
AxeCond_t *AxeGetRxMsgCond(void);

/**
 *	\brief		Get address of the AXE registers
 *				
 *	\return		Address of AXE registers
 *
 *	Returns the address in the current process where the AXE register
 *	have been mmapped.
 *
 */
void *AxeGetBaseAddr(void);


/**
 *	\brief		Map virtual address to physical address
 *	\param[in]	vPtr - virtual address 
 *				
 *	\return		Physical address of 'vPtr' parameter 
 *
 * 	Returns the physical address associated with virtual address
 * 	'vPtr'.  The parameter vPtr must be inside a region allocated
 * 	with AxeMalloc. 
 *
 */
void *AxeMapVirtToPhys(const void *vPtr);

/**
 *	\brief		Map physical address to virtual address
 *	\param[in]	pPtr - physical address 
 *				
 *	\return		Virtual address of 'pPtr' parameter 
 *
 * 	Returns the virtual address associated with physical address
 * 	'pPtr'.  The parameter pPtr must be inside a region allocated
 * 	with AxeMalloc. 
 *
 */
void *AxeMapPhysToVirt(const void *pPtr);


static inline int AxeMutexLock(AxeMutex_t *mutex)
{
	pthread_mutex_lock(mutex);
}

static inline int AxeMutexUnlock(AxeMutex_t *mutex)
{
	pthread_mutex_unlock(mutex);
}

int AxeMalloc(void **ptr, unsigned size, int region);
void AxeFree(void *ptr, unsigned size, int region);
void *AxeMapVirtToPhys(const void *vPtr);
void *AxeMapToPhysVirt(const void *pPtr);

#define AxeCondInit(cv) pthread_cond_init(cv, NULL)
#define AxeCondBroadcast pthread_cond_broadcast

typedef enum 
{
	AXE_OPT_TASK_STACK_WORDS,
	AXE_OPT_TASK_QUEUE_MESSAGES,
	AXE_OPT_DRIVER_QUEUE_MESSAGES,
	AXE_OPT_DRIVER_MSG_TIMEOUT
} AxeDriverOpts;

/**
 *	\brief		Return driver configuration option value
 *	\param[in]  opt - option to retrieve
 *	\param[out] value - 32-bit value of the option
 *
 *	This function returns one of a set of options that can be configured
 *	for the driver.  The value is a 32-bit unsigned integer but may 
 *	represent data of a different type (e.g. pointer to null-terminated
 *	character string).  It is the responsibility of the caller to 
 *	correctly typecast and interpret the value.
 *
 *  \note
 */
void AxeGetDriverOpt(AxeDriverOpts opt, unsigned *value);

/**
 *	\brief		Get a pointer to the RX message queue head pointer
 *				
 *	\return		Queue head pointer-pointer.
 *
 *	Accessor function for the shared RX message queue pointer. Must be used
 *	to allow this pointer to be shared across processes in some operating
 *	system environments.
 *
 *  \note		Since it is necessary to modify the head pointer, this
 *				function must return a pointer-pointer rather than simply
 *				the pointer to the first element in the queue.  Thus, this
 *				function will never return a NULL pointer.
 */
axeQueueMsg **AxeGetRxQueueHead(void);

/**
 *	\brief		Get a pointer to the TX message queue head pointer
 *				
 *	\return		Queue head pointer-pointer.
 *
 *	Accessor function for the shared TX message queue pointer. Must be used
 *	to allow this pointer to be shared across processes in some operating
 *	system environments.
 *
 *  \note		Since it is necessary to modify the head pointer, this
 *				function must return a pointer-pointer rather than simply
 *				the pointer to the first element in the queue.  Thus, this
 *				function will never return a NULL pointer.
 */
axeQueueMsg **AxeGetTxQueueHead(void);
